<?php

/**
 * StepNTest.php
 *
 * @since     2011-05-23
 * @category  Library
 * @package   Unicode
 * @author    Nicola Asuni <info@tecnick.com>
 * @copyright 2011-2024 Nicola Asuni - Tecnick.com LTD
 * @license   http://www.gnu.org/copyleft/lesser.html GNU-LGPL v3 (see LICENSE.TXT)
 * @link      https://github.com/tecnickcom/tc-lib-unicode
 *
 * This file is part of tc-lib-unicode software library.
 */

namespace Test\Bidi;

use Com\Tecnick\Unicode\Bidi\StepN;
use PHPUnit\Framework\TestCase;

/**
 * Bidi Test
 *
 * @since     2011-05-23
 * @category  Library
 * @package   Unicode
 * @author    Nicola Asuni <info@tecnick.com>
 * @copyright 2011-2024 Nicola Asuni - Tecnick.com LTD
 * @license   http://www.gnu.org/copyleft/lesser.html GNU-LGPL v3 (see LICENSE.TXT)
 * @link      https://github.com/tecnickcom/tc-lib-unicode
 */
class StepNTest extends TestCase
{
    /**
     * @dataProvider stepN0DataProvider
     *
     * @param array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        } $seq     Isolated Sequence array
     */
    public function testStepN0(array $seq, mixed $expected): void
    {
        $stepn = new StepN($seq, false);
        $stepn->processStep('getBracketPairs');
        $stepn->processStep('processN0');
        $this->assertEquals($expected, $stepn->getSequence());
    }

    /**
     * @return array<int,  array<int, array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        }>>
     */
    public static function stepN0DataProvider(): array
    {
        return [
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 0x3008,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 0x3009,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 0x3008,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 0x3009,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // ]
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 5,
                    'length' => 6,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // ]
                        [
                            'x' => 0,
                            'pos' => 5,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 5,
                    'length' => 6,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 5,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // ]
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // ]
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // ]
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        // [
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                    ],
                ],
                [
                    'e' => 1,
                    'edir' => 'R',
                    'start' => 0,
                    'end' => 4,
                    'length' => 5,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'R',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 91,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 93,
                            'i' => -1,
                            'level' => 1,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'ON',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 5,
                    'length' => 6,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 0x3008,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 0x3009,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 1809,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NSM',
                            'otype' => 'NSM',
                        ],
                        [
                            'x' => 0,
                            'pos' => 5,
                            'char' => 1809,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NSM',
                            'otype' => 'NSM',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 5,
                    'length' => 6,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 0x3008,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 0x3009,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 4,
                            'char' => 1809,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NSM',
                        ],
                        [
                            'x' => 0,
                            'pos' => 5,
                            'char' => 1809,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NSM',
                        ],
                    ],
                ],
            ],
        ];
    }

    /**
     * @dataProvider stepN1DataProvider
     *
     * @param array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        } $seq     Isolated Sequence array
     */
    public function testStepN1(array $seq, mixed $expected): void
    {
        $stepn = new \Com\Tecnick\Unicode\Bidi\StepN($seq, false);
        $stepn->processStep('processN1');
        $this->assertEquals($expected, $stepn->getSequence());
    }

    /**
     * @return array<int,  array<int, array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        }>>
     */
    public static function stepN1DataProvider(): array
    {
        return [
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1632,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'AN',
                            'otype' => 'AN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 1776,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'EN',
                            'otype' => 'EN',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 3,
                    'length' => 4,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 3,
                    'length' => 4,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 3,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
        ];
    }

    /**
     * @dataProvider stepN2DataProvider
     *
     * @param array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        } $seq     Isolated Sequence array
     */
    public function testStepN2(array $seq, mixed $expected): void
    {
        $stepn = new \Com\Tecnick\Unicode\Bidi\StepN($seq, false);
        $stepn->processStep('processN2');
        $this->assertEquals($expected, $stepn->getSequence());
    }

    /**
     * @return array<int,  array<int, array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        }>>
     */
    public static function stepN2DataProvider(): array
    {
        return [
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 1,
                    'length' => 1,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 1,
                    'length' => 1,
                    'maxlevel' => 0,
                    'sos' => 'L',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                    ],
                ],
            ],
        ];
    }

    /**
     * @dataProvider stepNDataProvider
     *
     * @param array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        } $seq     Isolated Sequence array
     */
    public function testStepN(array $seq, mixed $expected): void
    {
        $stepn = new \Com\Tecnick\Unicode\Bidi\StepN($seq);
        $this->assertEquals($expected, $stepn->getSequence());
    }

    /**
     * @return array<int,  array<int, array{
     *        'e': int,
     *        'edir': string,
     *        'end': int,
     *        'eos': string,
     *        'length': int,
     *        'maxlevel': int,
     *        'sos': string,
     *        'start': int,
     *        'item': array<int, array{
     *            'char': int,
     *            'i': int,
     *            'level': int,
     *            'otype': string,
     *            'pdimatch': int,
     *            'pos': int,
     *            'type': string,
     *            'x': int,
     *          }>,
     *        }>>
     */
    public static function stepNDataProvider(): array
    {
        return [
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 65,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'L',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 91,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'ON',
                            'otype' => 'ON',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                    ],
                ],
            ],
            [
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'NI',
                            'otype' => 'NI',
                        ],
                    ],
                ],
                [
                    'e' => 0,
                    'edir' => 'L',
                    'start' => 0,
                    'end' => 2,
                    'length' => 3,
                    'maxlevel' => 0,
                    'sos' => 'R',
                    'eos' => 'L',
                    'item' => [
                        [
                            'x' => 0,
                            'pos' => 0,
                            'char' => 8207,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'R',
                            'otype' => 'R',
                        ],
                        [
                            'x' => 0,
                            'pos' => 1,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                        [
                            'x' => 0,
                            'pos' => 2,
                            'char' => 5760,
                            'i' => -1,
                            'level' => 0,
                            'pdimatch' => -1,
                            'type' => 'L',
                            'otype' => 'NI',
                        ],
                    ],
                ],
            ],
        ];
    }
}
